import TSim.*;
import java.util.concurrent.*;
import java.util.*;


public class Lab1 {
    public static final int DIRECTION_UP = 1;
    public static final int DIRECTION_DOWN = -1;
    public static final int MAX_SPEED = 22;
    public static final int STATION_SLEEP_TIME_MS = 6000;
    public static final boolean IS_DEBUGGING = true;
    public static final boolean IS_DEBUGGING_INTER = false;
    public static Sensor[][] sensors = new Sensor[22][15];
    public static Map < Semaphore, String > mutexMap = new IdentityHashMap < Semaphore, String >(6);
    // Junction
    private static Semaphore J1 = new Semaphore(1);
    // Upper Station
    private static Semaphore S1 = new Semaphore(1);
    // Upper monorail
    private static Semaphore C1 = new Semaphore(1);
    // Fast track on parallell tracks
    private static Semaphore P1 = new Semaphore(1);
    // Lower monorail
    private static Semaphore C2 = new Semaphore(1);
    // Lower station
    private static Semaphore S2 = new Semaphore(1);

    private void createSensors() {
        // Mutex maps for sensible debug output
        mutexMap.put(J1, "J1");
        mutexMap.put(S1, "S1");
        mutexMap.put(C1, "C1");
        mutexMap.put(P1, "P1");
        mutexMap.put(C2, "C2");
        mutexMap.put(S2, "S2");

        // Station sensors
        addSensor(new StationSensor(15, 3, DIRECTION_UP));
        addSensor(new StationSensor(15, 5, DIRECTION_UP));
        addSensor(new StationSensor(15, 11, DIRECTION_DOWN));
        addSensor(new StationSensor(15, 13, DIRECTION_DOWN));

        // Sensors for rail crossing
        addSensor(new SingleRailSensor(6, 6, DIRECTION_DOWN, J1));
        addSensor(new SingleRailSensor(11, 7, DIRECTION_UP, J1));
        addSensor(new SingleRailSensor(8, 5, DIRECTION_DOWN, J1));
        addSensor(new SingleRailSensor(10, 8, DIRECTION_UP, J1));

        // Upper single track sensors
        addSensor(
                new SingleRailSensor(14, 7, DIRECTION_DOWN, C1,
                new Switch(17, 7, TSimInterface.SWITCH_RIGHT)));
        addSensor(
                new SingleRailSensor(15, 8, DIRECTION_DOWN, C1,
                new Switch(17, 7, TSimInterface.SWITCH_LEFT)));
        addSensor(
                new SingleRailSensor(12, 9, DIRECTION_UP, C1,
                new Switch(15, 9, TSimInterface.SWITCH_RIGHT)));
        addSensor(
                new SingleRailSensor(13, 10, DIRECTION_UP, C1,
                new Switch(15, 9, TSimInterface.SWITCH_LEFT)));

        // Lower single track sensors
        addSensor(
                new SingleRailSensor(6, 11, DIRECTION_UP, C2,
                new Switch(3, 11, TSimInterface.SWITCH_LEFT)));
        addSensor(
                new SingleRailSensor(4, 13, DIRECTION_UP, C2,
                new Switch(3, 11, TSimInterface.SWITCH_RIGHT)));
        addSensor(
                new SingleRailSensor(7, 9, DIRECTION_DOWN, C2,
                new Switch(4, 9, TSimInterface.SWITCH_LEFT)));
        addSensor(
                new SingleRailSensor(6, 10, DIRECTION_DOWN, C2,
                new Switch(4, 9, TSimInterface.SWITCH_RIGHT)));

        // Middle track selection
        addSensor(
                new ChooseRailSensor(2, 9, DIRECTION_UP, P1,
                new Switch(4, 9, TSimInterface.SWITCH_LEFT),
                new Switch(4, 9, TSimInterface.SWITCH_RIGHT)));
        addSensor(
                new ChooseRailSensor(17, 9, DIRECTION_DOWN, P1,
                new Switch(15, 9, TSimInterface.SWITCH_RIGHT),
                new Switch(15, 9, TSimInterface.SWITCH_LEFT)));
        
        // Station track selections
        addSensor(
                new ChooseRailSensor(19, 7, DIRECTION_UP, S1,
                new Switch(17, 7, TSimInterface.SWITCH_LEFT),
                new Switch(17, 7, TSimInterface.SWITCH_RIGHT)));
        addSensor(
                new ChooseRailSensor(1, 11, DIRECTION_DOWN, S2,
                new Switch(3, 11, TSimInterface.SWITCH_RIGHT),
                new Switch(3, 11, TSimInterface.SWITCH_LEFT)));

    }

    private static void addSensor(Sensor sensor) {
        int x = sensor.getXpos();
        int y = sensor.getYpos();

        sensors[x][y] = sensor;
    }

    public static void main(String[]args) {
        new Lab1(args);
    }

    public Lab1(String[]args) {
        int speeds[] = new int[2];

        try {
            speeds[0] = Integer.parseInt(args[0]);
            speeds[1] = Integer.parseInt(args[1]);
        } catch (NumberFormatException e) {
            System.err.println("Need 2 numerical arguments");
            System.exit(1);
        } catch (ArrayIndexOutOfBoundsException e) {
            System.err.println("Need 2 numerical arguments");
            System.exit(1);
        }

        for (int speed:speeds) {
            if (speed > MAX_SPEED || speed < 1) {
                System.err.println(
                        "Only speeds in the range: [1, " + MAX_SPEED
                        + "] allowed!");
                System.exit(1);
            }
        }

        TSimInterface inter = TSimInterface.getInstance();

        inter.setDebug(IS_DEBUGGING_INTER);

        createSensors();

        Thread train1 = new Thread(
                new Train(DIRECTION_DOWN, speeds[0], 1, inter));
        Thread train2 = new Thread(new Train(DIRECTION_UP, speeds[1], 2, inter));

        train1.start();
        train2.start();

        /*
         try {
         inter.setSpeed(1,10);
         inter.setSwitch(17,7, inter.SWITCH_RIGHT);
         }
         catch (CommandException e) {
         e.printStackTrace();    // or only e.getMessage() for the error
         System.exit(1);
         }
         */
    }

    public static void dbgPrint(String str) {
        if (IS_DEBUGGING) {
            System.err.println(str);
        }
    }

    public static void dbgPrint(String str, int trainID) {
        dbgPrint("Train " + trainID + ": " + str);
    }
}


class Switch {
    private int x_pos;
    private int y_pos;
    private int direction;

    public Switch(int x, int y, int dir) {
        x_pos = x;
        y_pos = y;
        direction = dir;
    }

    public void setSwitch(TSimInterface inter) {
        try {
            inter.setSwitch(x_pos, y_pos, direction);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}


abstract class Sensor {
    int x_pos;
    int y_pos;
    protected int direction;
    protected Semaphore[] rail_mutexes;
    protected Switch[] mySwitches;

    public Semaphore[] getRailMutexes() {
        return rail_mutexes;
    }

    public Switch[] getSwitches() {
        return mySwitches;
    }

    public int getXpos() {
        return x_pos;
    }

    public int getYpos() {
        return y_pos;
    }

    public Sensor(int x, int y, int dir) {
        x_pos = x;
        y_pos = y;
        direction = dir;
        Lab1.sensors[x][y] = this;
    }

    public boolean isMyDirection(int trainDirection) {
        return direction == trainDirection;
    }

    public void switchSwitches(TSim.TSimInterface inter) {
        for (Switch s:mySwitches) {
            s.setSwitch(inter);
        }
    }
}


class SingleRailSensor extends Sensor {
    public SingleRailSensor(int x, int y, int dir, Semaphore mutex) {
        super(x, y, dir);
        rail_mutexes = new Semaphore[] { mutex};
        mySwitches = new Switch[0];
    }

    public SingleRailSensor(int x, int y, int dir, Semaphore mutex, Switch mySwitch) {
        super(x, y, dir);
        rail_mutexes = new Semaphore[] { mutex};
        mySwitches = new Switch[] { mySwitch};
    }
}


/*
class RailWithConnectionSensor extends Sensor {
    // private Semaphore connection_mutex;
    public RailWithConnectionSensor(int x, int y, int dir, Semaphore rail_mutex,
            Switch rail_switch, Semaphore connection_mutex,
            Switch connection_switch) { 
        super(x, y, dir);
        // The order is VERY important
        // The correct order
        rail_mutexes = new Semaphore[] { connection_mutex, rail_mutex};
        
        // The wrong order:
        // rail_mutexes= new Semaphore[] {rail_mutex,connection_mutex};

        // For illustrating deadlock on wrong
        // order (try speeds 10, 19)

        mySwitches = new Switch[] { connection_switch, rail_switch};
    }
}
*/


class ChooseRailSensor extends Sensor {
    Switch[] altPath;
    public ChooseRailSensor(int x, int y, int dir, Semaphore mutex, Switch main_path, Switch alternative_path) {
        super(x, y, dir);
        rail_mutexes = new Semaphore[] { mutex};
        mySwitches = new Switch[] { main_path};
        altPath = new Switch[] { alternative_path};
    }

    public Switch[] getAltPathSwitches() {
        return altPath;
    }
}


class StationSensor extends Sensor {
    public StationSensor(int x, int y, int dir) {
        super(x, y, dir);
        // Do not acquire any mutexes
        rail_mutexes = new Semaphore[0];
        mySwitches = new Switch[0];
    }
}


class Train implements Runnable {
    private int current_direction;
    private int current_speed;
    private final int trainID;
    private ArrayList < Semaphore > mutexes_holded = new ArrayList < Semaphore >();
    private TSimInterface inter;

    public Train(int direction, int speed, int trainID, TSimInterface inter) {
        current_direction = direction;
        current_speed = speed;
        this.inter = inter;
        this.trainID = trainID;
        dbgPrint("Reporting");
    }

    private void dbgPrint(String str) {
        Lab1.dbgPrint(str, trainID);
    }

    private void stop_train() throws TSim.CommandException {
        inter.setSpeed(trainID, 0);
        dbgPrint("Stopping.");
    }

    private void start_train() throws TSim.CommandException {
        inter.setSpeed(trainID, current_speed);
        dbgPrint("Running. current_speed: " + current_speed);
    }

    private void onSensorEvent(Sensor sensor) throws TSim.CommandException, InterruptedException {
        Semaphore[] relevant_mutexes = sensor.getRailMutexes();

        if (sensor.isMyDirection(current_direction)) {
            if (sensor instanceof StationSensor) {
                stop_train();
                dbgPrint("Sleeping for " + Lab1.STATION_SLEEP_TIME_MS + " ms.");
                Thread.sleep(Lab1.STATION_SLEEP_TIME_MS);
                current_direction = -current_direction;
                current_speed = -current_speed;
                start_train();
            } else {

                Switch[]switches = sensor.getSwitches();

                for (Semaphore mutex:relevant_mutexes) {
                    dbgPrint("Getting mutex:" + Lab1.mutexMap.get(mutex));
                    if (mutex.tryAcquire()) {
                        mutexes_holded.add(mutex);
                    } else {
                        if (sensor instanceof ChooseRailSensor) {
                            // Choose other track
                            switches = ((ChooseRailSensor) sensor).getAltPathSwitches();
                        } else {
                            stop_train();
                            mutex.acquire();
                            mutexes_holded.add(mutex);
                            start_train();
                        }
                    }
                }
                dbgPrint("Setting switches");
                for (Switch s:switches) {
                    s.setSwitch(inter);
                }
            }
        } else {
            dbgPrint("Releasing mutexes");
            for (Semaphore mutex:relevant_mutexes) {
                dbgPrint("Releasing mutex:" + Lab1.mutexMap.get(mutex));
                if (mutexes_holded.contains(mutex)) {
                    mutex.release();
                    mutexes_holded.remove(mutex);
                }
            }
        }
    }

    public void run() {
        int xpos, ypos;

        try {
            start_train();
            while (true) {
                SensorEvent sensor_event = inter.getSensor(trainID);

                if (sensor_event.getStatus() == SensorEvent.ACTIVE) {
                    xpos = sensor_event.getXpos();
                    ypos = sensor_event.getYpos();
                    Sensor sensor = Lab1.sensors[xpos][ypos];

                    if (sensor != null) {
                        onSensorEvent(sensor);
                    } else {
                        dbgPrint("No known sensor at: " + xpos + ", " + ypos);
                    }
                }
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
	    System.exit(1);
        } catch (CommandException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

}
